/*
  ==============================================================================

    SonicCrypt Seq
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#include "PluginProcessor.h"
#include "PluginEditor.h"

static const int BASE_WIDTH = 1150;
static const int BASE_HEIGHT = 650;

NewProjectAudioProcessorEditor::NewProjectAudioProcessorEditor(NewProjectAudioProcessor& p)
    : AudioProcessorEditor(&p), audioProcessor(p)
{
    setResizable(true, true);
    setResizeLimits(1000, 600, 1920, 1080);

    setSize(audioProcessor.lastUIWidth, audioProcessor.lastUIHeight);

    getLookAndFeel().setColour(juce::Slider::thumbColourId, sonicAccent);
    getLookAndFeel().setColour(juce::Slider::rotarySliderFillColourId, sonicAccent);
    getLookAndFeel().setColour(juce::Slider::rotarySliderOutlineColourId, sonicHeader.brighter(0.1f));
    getLookAndFeel().setColour(juce::ComboBox::backgroundColourId, sonicHeader.brighter(0.1f));
    getLookAndFeel().setColour(juce::ComboBox::textColourId, sonicText);
    getLookAndFeel().setColour(juce::ComboBox::arrowColourId, sonicAccent);
    getLookAndFeel().setColour(juce::ToggleButton::tickColourId, sonicAccent);
    getLookAndFeel().setColour(juce::ToggleButton::tickDisabledColourId, sonicText.darker(0.5f));
    getLookAndFeel().setColour(juce::ToggleButton::textColourId, sonicText);

    auto logoImg = juce::ImageCache::getFromMemory(BinaryData::logo_png, BinaryData::logo_pngSize);
    if (logoImg.isValid()) {
        logoImage = logoImg;
        logoButton.setImages(false, true, true, logoImg, 1.0f, juce::Colours::transparentBlack, logoImg, 1.0f, juce::Colours::white.withAlpha(0.2f), logoImg, 1.0f, juce::Colours::transparentBlack);
    }
    logoButton.onClick = [] { juce::URL("https://www.soniccrypt.org").launchInDefaultBrowser(); };
    addAndMakeVisible(logoButton);

    addAndMakeVisible(presetGroup);
    presetGroup.setColour(juce::GroupComponent::outlineColourId, sonicHeader);
    presetGroup.setColour(juce::GroupComponent::textColourId, sonicText);

    addAndMakeVisible(presetListBox);
    presetListBox.setRowHeight(25);
    presetListBox.setColour(juce::ListBox::backgroundColourId, juce::Colours::black.withAlpha(0.3f));

    presetListModel = std::make_unique<PresetListComponent>(presetListBox, [this](juce::File file) {
        audioProcessor.loadPresetFile(file);
        updateUIFromData();
        repaint();
        });
    presetListBox.setModel(presetListModel.get());
    presetListModel->refresh(audioProcessor.getPresetFolder());

    addAndMakeVisible(savePresetButton);
    savePresetButton.setColour(juce::TextButton::buttonColourId, sonicAccent);
    savePresetButton.setColour(juce::TextButton::textColourOffId, sonicBgDark);
    savePresetButton.onClick = [this] { openSavePresetDialog(); };

    addAndMakeVisible(refreshButton);
    refreshButton.setColour(juce::TextButton::buttonColourId, sonicHeader);
    refreshButton.onClick = [this] { presetListModel->refresh(audioProcessor.getPresetFolder()); };

    addAndMakeVisible(copyrightBtn);
    copyrightBtn.setButtonText(juce::CharPointer_UTF8("\xc2\xa9 2025 SonicCrypt."));
    copyrightBtn.setColour(juce::TextButton::buttonColourId, juce::Colours::transparentBlack);
    copyrightBtn.setColour(juce::TextButton::textColourOffId, sonicText.darker(0.4f));
    copyrightBtn.onClick = [this] { aboutOverlay.setVisible(true); aboutOverlay.toFront(true); };

    auto setupLabel = [this](juce::Label& l) {
        l.setJustificationType(juce::Justification::centredLeft);
        l.setColour(juce::Label::textColourId, sonicText.darker(0.2f));
        };

    auto setupTab = [this](juce::TextButton& b, int idx) {
        b.setClickingTogglesState(true); b.setRadioGroupId(1001);
        b.setColour(juce::TextButton::buttonColourId, sonicHeader);
        b.setColour(juce::TextButton::buttonOnColourId, sonicAccent);
        b.setColour(juce::TextButton::textColourOnId, sonicBgDark);
        b.onClick = [this, idx] { selectLayer(idx); };
        addAndMakeVisible(b);
        };
    setupTab(layerBtnA, 0); layerBtnA.setToggleState(true, juce::dontSendNotification);
    setupTab(layerBtnB, 1); setupTab(layerBtnC, 2);

    // Mute Solo Init
    addAndMakeVisible(muteBtn);
    muteBtn.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.isMuted = muteBtn.getToggleState(); };
    addAndMakeVisible(soloBtn);
    soloBtn.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.isSoloed = soloBtn.getToggleState(); };

    addAndMakeVisible(layerActiveToggle);
    layerActiveToggle.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.isActive = layerActiveToggle.getToggleState(); };

    addAndMakeVisible(genAllBtn); genAllBtn.setColour(juce::TextButton::buttonColourId, sonicAccent); genAllBtn.setColour(juce::TextButton::textColourOffId, sonicBgDark); genAllBtn.onClick = [this] { randomizePattern(true); };
    addAndMakeVisible(genCurrBtn); genCurrBtn.setColour(juce::TextButton::buttonColourId, sonicHeader.brighter(0.2f)); genCurrBtn.setColour(juce::TextButton::textColourOffId, sonicText); genCurrBtn.onClick = [this] { randomizePattern(false); };
    addAndMakeVisible(chaosToggle);

    addAndMakeVisible(midiDragBtn);
    midiDragBtn.onDragStartCallback = [this]() {
        if (juce::DragAndDropContainer::isDragAndDropActive()) return;
        juce::File midiFile = audioProcessor.createMidiFileForExport(4);
        juce::StringArray files;
        files.add(midiFile.getFullPathName());
        performExternalDragDropOfFiles(files, true);
        };

    addAndMakeVisible(rateLabel); setupLabel(rateLabel); addAndMakeVisible(rateCombo); addAndMakeVisible(rateLock);
    rateCombo.addItem("1/4", 1); rateCombo.addItem("1/8", 2); rateCombo.addItem("1/16", 3); rateCombo.addItem("1/32", 4); rateCombo.addItem("1/8 T", 5); rateCombo.addItem("1/16 T", 6);
    rateCombo.onChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.rateIndex = rateCombo.getSelectedId() - 1; };
    rateLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.rateLocked = rateLock.getToggleState(); };

    addAndMakeVisible(rootLabel); setupLabel(rootLabel); addAndMakeVisible(rootCombo); addAndMakeVisible(rootLock);
    const char* noteNames[] = { "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B" };
    for (int i = 0; i < 12; ++i) rootCombo.addItem(noteNames[i], i + 1);
    rootCombo.onChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.rootNote = rootCombo.getSelectedId() - 1; };
    rootLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.rootLocked = rootLock.getToggleState(); };

    addAndMakeVisible(transLabel); setupLabel(transLabel); addAndMakeVisible(transSlider); addAndMakeVisible(transLock);
    transSlider.setSliderStyle(juce::Slider::IncDecButtons); transSlider.setRange(-24.0, 24.0, 1.0);
    transSlider.setColour(juce::Slider::textBoxTextColourId, sonicText); transSlider.setColour(juce::Slider::textBoxBackgroundColourId, sonicHeader.brighter(0.1f));
    transSlider.onValueChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.transpose = (int)transSlider.getValue(); };
    transLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.transLocked = transLock.getToggleState(); };

    addAndMakeVisible(scaleLabel); setupLabel(scaleLabel); addAndMakeVisible(scaleCombo); addAndMakeVisible(scaleLock);
    scaleCombo.addItem("Chromatic", 1); scaleCombo.addItem("Minor", 2); scaleCombo.addItem("Major", 3); scaleCombo.addItem("Pentatonic", 4); scaleCombo.addItem("Chord", 5);
    scaleCombo.onChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.scaleType = scaleCombo.getSelectedId() - 1; };
    scaleLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.scaleLocked = scaleLock.getToggleState(); };

    addAndMakeVisible(stepsLabel); setupLabel(stepsLabel); addAndMakeVisible(stepsSlider); addAndMakeVisible(stepsLock);
    stepsSlider.setSliderStyle(juce::Slider::IncDecButtons); stepsSlider.setRange(1.0, 32.0, 1.0);
    stepsSlider.setColour(juce::Slider::textBoxTextColourId, sonicText); stepsSlider.setColour(juce::Slider::textBoxBackgroundColourId, sonicHeader.brighter(0.1f));
    stepsSlider.onValueChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.numSteps = (int)stepsSlider.getValue(); repaint(); };
    stepsLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.stepsLocked = stepsLock.getToggleState(); };

    addAndMakeVisible(dirLabel); setupLabel(dirLabel); addAndMakeVisible(dirCombo); addAndMakeVisible(dirLock);
    dirCombo.addItem("Forward", 1); dirCombo.addItem("Backward", 2); dirCombo.addItem("Ping-Pong", 3); dirCombo.addItem("Random", 4);
    dirCombo.onChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.direction = dirCombo.getSelectedId() - 1; };
    dirLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.dirLocked = dirLock.getToggleState(); };

    addAndMakeVisible(octLabel); setupLabel(octLabel); addAndMakeVisible(octSlider); addAndMakeVisible(octLock);
    octSlider.setSliderStyle(juce::Slider::IncDecButtons); octSlider.setRange(1.0, 4.0, 1.0);
    octSlider.setColour(juce::Slider::textBoxTextColourId, sonicText); octSlider.setColour(juce::Slider::textBoxBackgroundColourId, sonicHeader.brighter(0.1f));
    octSlider.onValueChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.octaveRange = (int)octSlider.getValue(); };
    octLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.octLocked = octLock.getToggleState(); };

    addAndMakeVisible(gateLabel); setupLabel(gateLabel); gateLabel.setJustificationType(juce::Justification::centred);
    addAndMakeVisible(gateSlider); addAndMakeVisible(gateLock);
    gateSlider.setSliderStyle(juce::Slider::RotaryHorizontalVerticalDrag); gateSlider.setTextBoxStyle(juce::Slider::TextBoxBelow, false, 50, 20); gateSlider.setRange(0.1, 2.0, 0.01);
    gateSlider.setColour(juce::Slider::textBoxTextColourId, sonicText);
    gateSlider.onValueChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.gateLength = (float)gateSlider.getValue(); };
    gateLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.gateLocked = gateLock.getToggleState(); };

    addAndMakeVisible(swingLabel); setupLabel(swingLabel); swingLabel.setJustificationType(juce::Justification::centred);
    addAndMakeVisible(swingSlider); addAndMakeVisible(swingLock);
    swingSlider.setSliderStyle(juce::Slider::RotaryHorizontalVerticalDrag); swingSlider.setTextBoxStyle(juce::Slider::TextBoxBelow, false, 50, 20); swingSlider.setRange(0.0, 0.5, 0.01);
    swingSlider.setColour(juce::Slider::textBoxTextColourId, sonicText);
    swingSlider.onValueChange = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.swing = (float)swingSlider.getValue(); };
    swingLock.onClick = [this] { audioProcessor.getSequencerData().layers[currentLayerIndex].settings.swingLocked = swingLock.getToggleState(); };

    addAndMakeVisible(aboutOverlay); aboutOverlay.setVisible(false);
    updateUIFromData(); 
    startTimerHz(30);
    constructorFinished = true;
}

NewProjectAudioProcessorEditor::~NewProjectAudioProcessorEditor() { stopTimer(); }

void NewProjectAudioProcessorEditor::mouseWheelMove(const juce::MouseEvent& e, const juce::MouseWheelDetails& wheel)
{
    auto scrollCombo = [&](juce::ComboBox& combo) {
        if (combo.getBounds().contains(e.getPosition())) {
            int id = combo.getSelectedId();
            if (wheel.deltaY > 0) id++; else if (wheel.deltaY < 0) id--;
            if (id < 1) id = 1;
            if (id > combo.getNumItems()) id = combo.getNumItems();
            combo.setSelectedId(id, juce::sendNotification);
        }
        };
    scrollCombo(rateCombo);
    scrollCombo(scaleCombo);
    scrollCombo(dirCombo);
    scrollCombo(rootCombo);
}

void NewProjectAudioProcessorEditor::paint(juce::Graphics& g)
{
    g.fillAll(sonicBgDark);

    float scale = (float)getWidth() / (float)BASE_WIDTH;
    int sidebarW = (int)(220 * scale);
    int mainX = sidebarW + (int)(10 * scale);
    int headerHeight = (int)(260 * scale);

    g.setColour(sonicHeader.darker(0.2f));
    g.fillRect(0, 0, sidebarW, getHeight());
    g.setColour(sonicAccent.withAlpha(0.2f)); g.drawVerticalLine(sidebarW, 0.0f, (float)getHeight());

    auto mainArea = getLocalBounds().withTrimmedLeft(mainX);
    auto controlArea = mainArea.removeFromTop(headerHeight);
    auto sequencerArea = mainArea.reduced((int)(10 * scale));

    g.setColour(sonicHeader); g.fillRect(controlArea);
    g.setColour(sonicAccent.withAlpha(0.3f)); g.drawRect(controlArea.toFloat(), 1.0f);

    auto& layer = audioProcessor.getSequencerData().layers[currentLayerIndex];
    int numSteps = layer.settings.numSteps; if (numSteps < 1) numSteps = 1;
    float stepWidth = (float)sequencerArea.getWidth() / (float)numSteps;
    float height = (float)sequencerArea.getHeight();
    int playStep = audioProcessor.getCurrentStepForLayer(currentLayerIndex);

    for (int i = 0; i < numSteps; ++i) {
        juce::Rectangle<float> stepRect(sequencerArea.getX() + i * stepWidth, (float)sequencerArea.getY(), stepWidth - (2.0f * scale), height);
        bool isActive = layer.steps[i].active; int noteOffset = layer.steps[i].noteOffset;
        juce::Colour stepColor = isActive ? sonicAccent : sonicHeader.brighter(0.2f);
        if (i == playStep) stepColor = juce::Colours::white;

        g.setColour(stepColor.withAlpha(isActive ? 0.3f : 0.1f)); g.fillRect(stepRect);
        float normalizedPitch = juce::jlimit(0.0f, 1.0f, juce::jmap((float)noteOffset, -24.0f, 24.0f, 0.0f, 1.0f));
        float barHeight = stepRect.getHeight() * normalizedPitch;
        float yPos = stepRect.getBottom() - barHeight;
        if (barHeight < 4.0f) { barHeight = 4.0f; yPos = stepRect.getBottom() - 4.0f; }

        g.setColour(stepColor); g.fillRect(stepRect.getX(), yPos, stepRect.getWidth(), barHeight);
        g.setColour(sonicBgDark); g.drawRect(stepRect, 1.0f);
        if (i % 4 == 0 && i > 0) {
            g.setColour(sonicText.withAlpha(0.2f));
            g.drawVerticalLine((int)stepRect.getX() - 1, (float)sequencerArea.getY(), (float)sequencerArea.getBottom());
        }
    }
}

void NewProjectAudioProcessorEditor::resized()
{
    // Sicherheitscheck: Nur speichern, wenn die Größe valide ist (> 100px)
    if (constructorFinished) {
        if (getWidth() > 100 && getHeight() > 100) {
            audioProcessor.lastUIWidth = getWidth();
            audioProcessor.lastUIHeight = getHeight();
        }
    }

    float scale = (float)getWidth() / (float)BASE_WIDTH;
    updateFontSize();

    // SIDEBAR
    int sidebarW = (int)(220 * scale);
    auto sidebar = getLocalBounds().removeFromLeft(sidebarW).reduced((int)(10 * scale));

    logoButton.setBounds(sidebar.removeFromTop((int)(70 * scale)));
    copyrightBtn.setBounds(sidebar.removeFromTop((int)(25 * scale)));
    sidebar.removeFromTop(20);

    presetGroup.setBounds(sidebar);
    auto listArea = sidebar.reduced(10, 25);
    int btnH = (int)(30 * scale);

    savePresetButton.setBounds(listArea.removeFromBottom(btnH).removeFromLeft(listArea.getWidth() / 2).reduced(2));
    refreshButton.setBounds(listArea.removeFromBottom(btnH).reduced(2));
    refreshButton.setBounds(savePresetButton.getRight() + 4, savePresetButton.getY(), savePresetButton.getWidth(), savePresetButton.getHeight());

    listArea.removeFromBottom(5);
    presetListBox.setBounds(listArea);

    // MAIN AREA
    int mainX = sidebarW + (int)(10 * scale);
    auto mainArea = getLocalBounds().withTrimmedLeft(mainX);
    int headerHeight = (int)(260 * scale);
    auto topBar = mainArea.removeFromTop(headerHeight).reduced((int)(5 * scale));

    // Tabs
    auto tabRow = topBar.removeFromTop((int)(30 * scale));
    int tabW = (int)(80 * scale);
    layerBtnA.setBounds(tabRow.removeFromLeft(tabW).reduced(2));
    layerBtnB.setBounds(tabRow.removeFromLeft(tabW).reduced(2));
    layerBtnC.setBounds(tabRow.removeFromLeft(tabW).reduced(2));

    // Mute / Solo
    int smallBtn = (int)(25 * scale);
    muteBtn.setBounds(tabRow.removeFromLeft(smallBtn).reduced(2));
    soloBtn.setBounds(tabRow.removeFromLeft(smallBtn).reduced(2));

    layerActiveToggle.setBounds(tabRow.removeFromLeft((int)(80 * scale)).reduced(5, 0));

    topBar.removeFromTop(5);

    auto leftCol = topBar.removeFromLeft((int)(mainArea.getWidth() * 0.4f));
    auto rightCol = topBar;

    int rowH = leftCol.getHeight() / 7;

    auto layoutRow = [&](juce::Rectangle<int>& area, juce::Label& lbl, juce::Component& ctrl, juce::Button& lck) {
        int lW = (int)(50 * scale); int kW = (int)(25 * scale);
        lbl.setBounds(area.removeFromLeft(lW));
        lck.setBounds(area.removeFromRight(kW).withSizeKeepingCentre((int)(20 * scale), (int)(20 * scale)));
        ctrl.setBounds(area.reduced(2, 2));
        };
    auto layoutSliderRow = [&](juce::Rectangle<int>& area, juce::Label& lbl, juce::Slider& sld, juce::Button& lck) {
        int lW = (int)(50 * scale); int kW = (int)(25 * scale);
        lbl.setBounds(area.removeFromLeft(lW));
        lck.setBounds(area.removeFromRight(kW).withSizeKeepingCentre((int)(20 * scale), (int)(20 * scale)));
        int tbW = (int)(60 * scale); int tbH = (int)(20 * scale);
        sld.setTextBoxStyle(juce::Slider::TextBoxLeft, false, tbW, tbH);
        sld.setBounds(area.reduced(2, 2));
        };

    auto r1 = leftCol.removeFromTop(rowH); layoutRow(r1, rateLabel, rateCombo, rateLock);
    auto r2 = leftCol.removeFromTop(rowH); layoutRow(r2, rootLabel, rootCombo, rootLock);
    auto r3 = leftCol.removeFromTop(rowH); layoutSliderRow(r3, transLabel, transSlider, transLock);
    auto r4 = leftCol.removeFromTop(rowH); layoutRow(r4, scaleLabel, scaleCombo, scaleLock);
    auto r5 = leftCol.removeFromTop(rowH); layoutSliderRow(r5, stepsLabel, stepsSlider, stepsLock);
    auto r6 = leftCol.removeFromTop(rowH); layoutRow(r6, dirLabel, dirCombo, dirLock);
    auto r7 = leftCol.removeFromTop(rowH); layoutSliderRow(r7, octLabel, octSlider, octLock);

    // Right Col
    auto upperRight = rightCol.removeFromTop((int)(rowH * 3.5));
    int knobW = upperRight.getWidth() / 2;
    auto gateZone = upperRight.removeFromLeft(knobW);
    int knobSize = (int)(90 * scale);

    auto gateCont = gateZone.withSizeKeepingCentre(knobSize, (int)(80 * scale));
    gateLabel.setBounds(gateCont.removeFromTop((int)(20 * scale)));
    gateLock.setBounds(gateCont.removeFromRight((int)(25 * scale)).withSizeKeepingCentre((int)(20 * scale), (int)(20 * scale)));
    gateSlider.setBounds(gateCont);

    auto swingCont = upperRight.withSizeKeepingCentre(knobSize, (int)(80 * scale));
    swingLabel.setBounds(swingCont.removeFromTop((int)(20 * scale)));
    swingLock.setBounds(swingCont.removeFromRight((int)(25 * scale)).withSizeKeepingCentre((int)(20 * scale), (int)(20 * scale)));
    swingSlider.setBounds(swingCont);

    auto chaosRow = rightCol.removeFromTop(rowH);
    chaosToggle.setBounds(chaosRow.withSizeKeepingCentre((int)(140 * scale), (int)(20 * scale)));

    // Gen & Drag Row
    auto genRow = rightCol;
    auto genCont = genRow.withSizeKeepingCentre((int)(300 * scale), (int)(30 * scale));

    genAllBtn.setBounds(genCont.removeFromLeft(80).reduced(2));
    genCurrBtn.setBounds(genCont.removeFromLeft(80).reduced(2));
    midiDragBtn.setBounds(genCont.reduced(2));

    aboutOverlay.setBounds(getLocalBounds());
}

void NewProjectAudioProcessorEditor::updateFontSize()
{
    float scale = (float)getWidth() / (float)BASE_WIDTH;
    float fontSize = juce::jlimit(12.0f, 18.0f, 14.0f * scale);
    juce::FontOptions fo(fontSize, juce::Font::bold);
    rateLabel.setFont(fo); scaleLabel.setFont(fo); stepsLabel.setFont(fo); rootLabel.setFont(fo); transLabel.setFont(fo);
    dirLabel.setFont(fo); octLabel.setFont(fo); gateLabel.setFont(fo); swingLabel.setFont(fo);
}

void NewProjectAudioProcessorEditor::selectLayer(int index) { currentLayerIndex = index; layerBtnA.setToggleState(index == 0, juce::dontSendNotification); layerBtnB.setToggleState(index == 1, juce::dontSendNotification); layerBtnC.setToggleState(index == 2, juce::dontSendNotification); updateUIFromData(); repaint(); }

void NewProjectAudioProcessorEditor::updateUIFromData() {
    auto& s = audioProcessor.getSequencerData().layers[currentLayerIndex].settings;
    layerActiveToggle.setToggleState(s.isActive, juce::dontSendNotification);
    muteBtn.setToggleState(s.isMuted, juce::dontSendNotification);
    soloBtn.setToggleState(s.isSoloed, juce::dontSendNotification);

    rateCombo.setSelectedId(s.rateIndex + 1, juce::dontSendNotification);
    rootCombo.setSelectedId(s.rootNote + 1, juce::dontSendNotification);
    transSlider.setValue(s.transpose, juce::dontSendNotification);
    scaleCombo.setSelectedId(s.scaleType + 1, juce::dontSendNotification);
    stepsSlider.setValue(s.numSteps, juce::dontSendNotification);
    dirCombo.setSelectedId(s.direction + 1, juce::dontSendNotification);
    octSlider.setValue(s.octaveRange, juce::dontSendNotification);
    swingSlider.setValue(s.swing, juce::dontSendNotification);
    gateSlider.setValue(s.gateLength, juce::dontSendNotification);

    rateLock.setToggleState(s.rateLocked, juce::dontSendNotification);
    rootLock.setToggleState(s.rootLocked, juce::dontSendNotification);
    transLock.setToggleState(s.transLocked, juce::dontSendNotification);
    scaleLock.setToggleState(s.scaleLocked, juce::dontSendNotification);
    stepsLock.setToggleState(s.stepsLocked, juce::dontSendNotification);
    dirLock.setToggleState(s.dirLocked, juce::dontSendNotification);
    octLock.setToggleState(s.octLocked, juce::dontSendNotification);
    gateLock.setToggleState(s.gateLocked, juce::dontSendNotification);
    swingLock.setToggleState(s.swingLocked, juce::dontSendNotification);
}

void NewProjectAudioProcessorEditor::randomizePattern(bool allLayers) {
    juce::Random r;
    bool isChaos = chaosToggle.getToggleState();
    int startIdx = allLayers ? 0 : currentLayerIndex;
    int endIdx = allLayers ? 3 : currentLayerIndex + 1;

    for (int i = startIdx; i < endIdx; ++i) {
        auto& layer = audioProcessor.getSequencerData().layers[i];
        auto& s = layer.settings;

        if (i == currentLayerIndex) {
            s.rateLocked = rateLock.getToggleState(); s.scaleLocked = scaleLock.getToggleState();
            s.rootLocked = rootLock.getToggleState();
            s.transLocked = transLock.getToggleState();
            s.stepsLocked = stepsLock.getToggleState(); s.dirLocked = dirLock.getToggleState();
            s.octLocked = octLock.getToggleState(); s.gateLocked = gateLock.getToggleState();
            s.swingLocked = swingLock.getToggleState();
        }

        float oGate = s.gateLength; float oSwing = s.swing;
        int oRate = s.rateIndex; int oScale = s.scaleType; int oRoot = s.rootNote;
        int oTrans = s.transpose;
        int oSteps = s.numSteps; int oDir = s.direction; int oOct = s.octaveRange;

        for (auto& step : layer.steps) { step.active = false; step.noteOffset = 0; step.velocity = 0.8f; }

        if (!s.gateLocked) s.gateLength = r.nextFloat() * 0.4f + 0.3f; else s.gateLength = oGate;
        if (!s.swingLocked) s.swing = (r.nextFloat() > 0.5f) ? r.nextFloat() * 0.25f : 0.0f; else s.swing = oSwing;
        if (!s.rateLocked) s.rateIndex = r.nextInt(4); else s.rateIndex = oRate;
        if (!s.stepsLocked) s.numSteps = r.nextInt({ 8, 33 }); else s.numSteps = oSteps;
        if (!s.dirLocked) s.direction = (r.nextFloat() > 0.8f) ? r.nextInt(4) : 0; else s.direction = oDir;
        if (!s.octLocked) s.octaveRange = (r.nextFloat() > 0.7f) ? 2 : 1; else s.octaveRange = oOct;
        if (!s.scaleLocked) s.scaleType = r.nextInt(5); else s.scaleType = oScale;
        if (!s.rootLocked) s.rootNote = r.nextInt(12); else s.rootNote = oRoot;
        if (!s.transLocked) s.transpose = (r.nextFloat() > 0.8f) ? (r.nextInt(3) - 1) * 12 : 0; else s.transpose = oTrans;

        if (isChaos) {
            for (int k = 0; k < 32; ++k) {
                float prob = (k % 4 == 0) ? 0.9f : 0.4f;
                if (r.nextFloat() < prob) {
                    layer.steps[k].active = true;
                    layer.steps[k].noteOffset = r.nextInt({ -24, 25 });
                    layer.steps[k].velocity = juce::jlimit(0.1f, 1.0f, r.nextFloat() * 0.4f + 0.6f);
                }
            }
        }
        else {
            // Euclidean Generator
            int steps = s.numSteps;
            int hits = r.nextInt(juce::Range<int>(steps / 4, steps));
            int shift = r.nextInt(steps);
            auto scaleNotes = SequencerData::getScaleIntervals(s.scaleType);

            for (int k = 0; k < steps; ++k) {
                int bucket = ((k + shift) * hits) % steps;
                bool isHit = bucket < hits;

                if (isHit) {
                    layer.steps[k].active = true;
                    int nIdx = r.nextInt(scaleNotes.size());
                    int noteVal = scaleNotes[nIdx];
                    if (k == 0) noteVal = scaleNotes[0];
                    layer.steps[k].noteOffset = noteVal;

                    float baseVel = (k % 4 == 0) ? 1.0f : 0.75f;
                    float humanJitter = (r.nextFloat() - 0.5f) * 0.2f;
                    layer.steps[k].velocity = juce::jlimit(0.1f, 1.0f, baseVel + humanJitter);
                }
            }
        }
    }
    updateUIFromData();
    repaint();
}

void NewProjectAudioProcessorEditor::openSavePresetDialog() {
    fileChooser = std::make_shared<juce::FileChooser>("Save Multi-Layer Preset", audioProcessor.getPresetFolder(), "*.xml");
    fileChooser->launchAsync(juce::FileBrowserComponent::saveMode | juce::FileBrowserComponent::canSelectFiles, [this](const juce::FileChooser& fc) {
        auto file = fc.getResult();
        if (file != juce::File{}) {
            if (!file.hasFileExtension("xml")) file = file.withFileExtension("xml");
            audioProcessor.savePreset(file.getFileNameWithoutExtension());
            presetListModel->refresh(audioProcessor.getPresetFolder());
        }
        });
}

void NewProjectAudioProcessorEditor::mouseDown(const juce::MouseEvent& e) {
    float scale = (float)getWidth() / (float)BASE_WIDTH;
    int sidebarW = (int)(220 * scale);
    int mainX = sidebarW + (int)(10 * scale);
    int headerHeight = (int)(260 * scale);

    if (e.x < mainX || e.y < headerHeight) return;

    auto seqBounds = getLocalBounds().withTrimmedLeft(mainX);
    seqBounds.removeFromTop(headerHeight); seqBounds.reduce(10, 10);

    auto& layer = audioProcessor.getSequencerData().layers[currentLayerIndex];
    int numSteps = layer.settings.numSteps;
    float stepWidth = (float)seqBounds.getWidth() / (float)numSteps;

    int idx = (int)((e.x - seqBounds.getX()) / stepWidth);
    if (idx >= 0 && idx < numSteps) { layer.steps[idx].active = !layer.steps[idx].active; repaint(); }
}
void NewProjectAudioProcessorEditor::mouseDrag(const juce::MouseEvent& e) {
    float scale = (float)getWidth() / (float)BASE_WIDTH;
    int sidebarW = (int)(220 * scale);
    int mainX = sidebarW + (int)(10 * scale);
    int headerHeight = (int)(260 * scale);

    if (e.x < mainX || e.y < headerHeight) return;

    auto seqBounds = getLocalBounds().withTrimmedLeft(mainX);
    seqBounds.removeFromTop(headerHeight); seqBounds.reduce(10, 10);

    auto& layer = audioProcessor.getSequencerData().layers[currentLayerIndex];
    int numSteps = layer.settings.numSteps;
    float stepWidth = (float)seqBounds.getWidth() / (float)numSteps;

    int idx = (int)((e.x - seqBounds.getX()) / stepWidth);
    if (idx >= 0 && idx < numSteps) {
        float h = (float)seqBounds.getHeight();
        float nY = 1.0f - ((float)(e.y - seqBounds.getY()) / h);
        layer.steps[idx].noteOffset = juce::jlimit(-24, 24, (int)juce::jmap(nY, 0.0f, 1.0f, -24.0f, 24.0f));
        if (!layer.steps[idx].active) layer.steps[idx].active = true;
        repaint();
    }
}
void NewProjectAudioProcessorEditor::timerCallback() { bool c = chaosToggle.getToggleState(); juce::String s = c ? " (CHAOS)" : ""; genAllBtn.setButtonText("GEN ALL" + s); juce::String ln = (currentLayerIndex == 0 ? "A" : (currentLayerIndex == 1 ? "B" : "C")); genCurrBtn.setButtonText("GEN " + ln + s); repaint(); }